# Cloudflare Bot Infrastructure Setup

Complete toolkit for bots to create shareable content: Quick Tunnels, R2 storage, and Pages hosting.

## What's Included

1. **Quick Tunnels** - Expose local ports instantly (no account needed)
2. **R2 File Storage** - Zero-egress object storage with public URLs
3. **Pages Hosting** - Deploy static sites to CDN

## Quick Start

### Prerequisites

```bash
# Install Cloudflare Wrangler CLI
npm install -g wrangler

# Login to Cloudflare
wrangler login
```

### Installation

1. Copy the `cloudflare/` directory to your workspace
2. Make scripts executable:
```bash
chmod +x cloudflare/*.sh
```

3. Set environment variables (optional - can also use `wrangler login`):
```bash
export CLOUDFLARE_ACCOUNT_ID="your-account-id"
export CLOUDFLARE_API_TOKEN="your-api-token"
```

## Usage

### Quick Tunnels (No Account Needed)

Expose local ports with temporary public URLs:

```bash
# Start a tunnel
./cloudflare/start-tunnel.sh 3000
# Returns: https://random-name.trycloudflare.com

# Stop tunnel
./cloudflare/stop-tunnel.sh
```

**Example - Share a file:**
```bash
python3 -m http.server 8080 --directory /tmp &
URL=$(./cloudflare/start-tunnel.sh 8080)
echo "Download: $URL/myfile.pdf"
./cloudflare/stop-tunnel.sh  # when done
```

### R2 Object Storage

Upload files and get permanent public URLs:

```bash
# Upload a file
./cloudflare/upload-r2.sh /path/to/file.pdf

# Or with custom name
./cloudflare/upload-r2.sh /path/to/file.pdf custom-name.pdf
```

**Setup R2 bucket (one-time):**
```bash
# Create bucket
wrangler r2 bucket create your-bucket-name

# Enable public access
# Go to dashboard → R2 → your-bucket → Settings → Public Access → Enable
```

**Note:** Update `upload-r2.sh` with your bucket name.

### Pages Hosting

Deploy static HTML sites to Cloudflare CDN:

```bash
# Deploy a directory
./cloudflare/deploy-page.sh /path/to/html my-project-name
```

**Example - Deploy a report:**
```bash
mkdir -p /tmp/report
cat > /tmp/report/index.html << 'EOF'
<!DOCTYPE html>
<html>
<head><title>My Report</title></head>
<body><h1>Analysis Report</h1></body>
</html>
EOF

./cloudflare/deploy-page.sh /tmp/report analysis-feb2026
# Returns: https://analysis-feb2026.pages.dev
```

## Files Included

```
cloudflare/
├── README.md           # Quick reference
├── start-tunnel.sh     # Start Quick Tunnel
├── stop-tunnel.sh      # Stop Quick Tunnel
├── upload-r2.sh        # Upload to R2 storage
└── deploy-page.sh      # Deploy to Pages
```

## Configuration

### Update R2 Bucket Name

Edit `upload-r2.sh`:
```bash
BUCKET="your-bucket-name"
```

### Update Account Details

If not using `wrangler login`, set these in your environment or in the scripts:

```bash
export CLOUDFLARE_ACCOUNT_ID="your-account-id"
export CLOUDFLARE_API_TOKEN="your-api-token"
```

## Common Patterns

### Bot Intelligence Reports

```bash
# Generate JSON data
echo '{"title": "Competitor Analysis", "data": [...]}' > /tmp/data.json

# Create HTML from JSON
# ... your report generation code ...

# Deploy to Pages
./cloudflare/deploy-page.sh /tmp/report competitor-analysis
```

### Temporary File Sharing

```bash
# Quick share (ephemeral)
python3 -m http.server 8080 --directory /tmp &
URL=$(./cloudflare/start-tunnel.sh 8080)
echo "Share: $URL/file.zip"

# Permanent share (R2)
./cloudflare/upload-r2.sh /tmp/file.zip
```

### Multi-File Sharing

```bash
# Create index
mkdir -p /tmp/share
cd /tmp/share
ls -1 *.pdf | while read f; do
  echo "<a href='$f'>$f</a><br>" >> index.html
done

# Share via tunnel
URL=$(./cloudflare/start-tunnel.sh 8080)
echo "Browse: $URL"
```

## Troubleshooting

### "Command not found: wrangler"
```bash
npm install -g wrangler
```

### "Authentication error"
```bash
wrangler login
# Or set CLOUDFLARE_API_TOKEN
```

### R2 uploads return 404
- Public access may still be propagating (wait 2-3 minutes)
- Verify public access is enabled in dashboard

### Pages deployment fails
- Ensure directory contains `index.html` or valid static site
- Check project name is unique (lowercase, hyphens only)

## Costs

All three systems are **free tier eligible**:

- **Quick Tunnels**: Free, no account needed
- **R2**: 10GB storage + unlimited egress (free)
- **Pages**: 500 builds/month + unlimited bandwidth (free)

## Security Notes

- **Quick Tunnels**: Temporary URLs, expire when tunnel stops
- **R2**: Public URLs are permanent until you delete the file
- **Pages**: URLs are permanent, support custom domains

For private sharing, use Quick Tunnels (ephemeral) instead of R2/Pages.

## References

- [Cloudflare Wrangler Docs](https://developers.cloudflare.com/workers/wrangler/)
- [R2 Storage Docs](https://developers.cloudflare.com/r2/)
- [Pages Hosting Docs](https://developers.cloudflare.com/pages/)
- [Cloudflare Tunnels](https://developers.cloudflare.com/cloudflare-one/connections/connect-apps/)

---

Created: 2026-02-28  
Version: 1.0.0
